<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../config.php';

if (!isset($_SESSION['portal_user'])) {
    header("Location: index.php?page=login");
    exit;
}

$db = DB::get();

// Oturumdaki ID ile en güncel kullanıcıyı DB'den çek
$sessionUser = $_SESSION['portal_user'];
$userId      = (int)$sessionUser['id'];

$stmt = $db->prepare("SELECT * FROM portal_users WHERE id = ?");
$stmt->execute([$userId]);
$user = $stmt->fetch();

if (!$user) {
    // Kullanıcı bulunamazsa güvenli çık
    unset($_SESSION['portal_user']);
    header("Location: index.php?page=login");
    exit;
}

// Oturumla senkron tut
$_SESSION['portal_user'] = $user;

$message = '';
$success = '';

/**
 * Küçük helper: profil değişikliği kaydı oluştur
 * field: username | email | avatar
 */
function addProfileChange(PDO $db, int $userId, string $field, string $oldValue, string $newValue)
{
    try {
        $stmt = $db->prepare("
            INSERT INTO portal_profile_changes
                (user_id, field, old_value, new_value, status, admin_note, created_at)
            VALUES
                (?, ?, ?, ?, 'pending', '', NOW())
        ");
        $stmt->execute([$userId, $field, $oldValue, $newValue]);
    } catch (Exception $e) {
        // Tablo henüz yoksa vs. hata verirse sessiz geçiyoruz
        // (gerekirse loglanabilir)
    }
}

/* -------------------------------------------------------------------------- */
/*                           AYARLAR (HESAP + AVATAR)                         */
/* -------------------------------------------------------------------------- */

if (isset($_POST['save_settings'])) {

    $targetUsername = trim($_POST['username'] ?? '');
    $targetEmail    = trim($_POST['email'] ?? '');

    $oldPass  = $_POST['old_password']   ?? '';
    $newPass  = $_POST['new_password']   ?? '';
    $newPass2 = $_POST['new_password2']  ?? '';

    $passwordUpdated   = false;
    $changeRequestSent = false;

    // --------- Temel validasyonlar ---------
    if (strlen($targetUsername) < 3) {
        $message = "Kullanıcı adı en az 3 karakter olmalıdır.";
    }

    if (!filter_var($targetEmail, FILTER_VALIDATE_EMAIL)) {
        $message = "Geçerli bir e-posta adresi girin.";
    } else {
        // Aynı e-posta başka bir kullanıcıda mı?
        $check = $db->prepare("SELECT id FROM portal_users WHERE email = ? AND id != ?");
        $check->execute([$targetEmail, $userId]);
        if ($check->rowCount() > 0) {
            $message = "Bu e-posta başka bir kullanıcı tarafından kullanılıyor.";
        }
    }

    // --------- Şifre değiştirme (anında uygulanır) ---------
    if (!$message && ($oldPass || $newPass || $newPass2)) {

        $stmt = $db->prepare("SELECT password_hash FROM portal_users WHERE id = ?");
        $stmt->execute([$userId]);
        $row = $stmt->fetch();

        if (!password_verify($oldPass, $row['password_hash'] ?? '')) {
            $message = "Eski şifre hatalı!";
        } elseif ($newPass !== $newPass2) {
            $message = "Yeni şifreler eşleşmiyor!";
        } elseif (strlen($newPass) < 5) {
            $message = "Yeni şifre en az 5 karakter olmalıdır!";
        } else {
            $newHash = password_hash($newPass, PASSWORD_DEFAULT);
            $stmt    = $db->prepare("UPDATE portal_users SET password_hash = ? WHERE id = ?");
            $stmt->execute([$newHash, $userId]);
            $passwordUpdated = true;
        }
    }

    // --------- Avatar yükleme (admin onayına gider) ---------
    $pendingAvatarUrl = null;

    if (!$message && isset($_FILES['avatar']) && $_FILES['avatar']['error'] !== UPLOAD_ERR_NO_FILE) {

        if ($_FILES['avatar']['error'] !== UPLOAD_ERR_OK) {
            $message = '❌ Avatar yüklenirken bir hata oluştu.';
        } else {

            $tmp  = $_FILES['avatar']['tmp_name'];
            $name = $_FILES['avatar']['name'];
            $size = $_FILES['avatar']['size'];

            $ext      = strtolower(pathinfo($name, PATHINFO_EXTENSION));
            $allowed  = ['png', 'jpg', 'jpeg'];
            $maxSize  = 2 * 1024 * 1024; // 2MB

            if (!in_array($ext, $allowed, true)) {
                $message = '❌ Sadece PNG, JPG, JPEG yükleyebilirsin.';
            } elseif ($size > $maxSize) {
                $message = '❌ Maksimum dosya boyutu 2MB.';
            } else {
                // Onay bekleyen avatarı farklı bir isimle kaydet
                $newName = 'pending_avatar_' . $userId . '_' . time() . '.' . $ext;
                $path    = AVATAR_UPLOAD_DIR . $newName;

                if (move_uploaded_file($tmp, $path)) {
                    $pendingAvatarUrl = AVATAR_URL . $newName;
                } else {
                    $message = '❌ Dosya yükleme hatası.';
                }
            }
        }
    }

    // --------- Kullanıcı adı / e-posta / avatar isteklerini kaydet ---------
    if (!$message) {

        // Kullanıcı adı değiştiyse admin onayına gönder
        if ($targetUsername !== '' && $targetUsername !== $user['username']) {
            addProfileChange($db, $userId, 'username', $user['username'], $targetUsername);
            $changeRequestSent = true;
        }

        // E-posta değiştiyse admin onayına gönder
        if ($targetEmail !== '' && $targetEmail !== $user['email']) {
            addProfileChange($db, $userId, 'email', $user['email'], $targetEmail);
            $changeRequestSent = true;
        }

        // Avatar yüklendiyse admin onayına gönder
        if ($pendingAvatarUrl !== null) {
            $oldAvatar = $user['avatar'] ?? '';
            addProfileChange($db, $userId, 'avatar', $oldAvatar, $pendingAvatarUrl);
            $changeRequestSent = true;
        }

        // Kullanıcıya özet mesaj
        if ($passwordUpdated && $changeRequestSent) {
            $success = "Şifren güncellendi. Diğer profil değişikliklerin yönetici onayına gönderildi.";
        } elseif ($passwordUpdated) {
            $success = "Şifren başarıyla güncellendi.";
        } elseif ($changeRequestSent) {
            $success = "Profil değişikliklerin yönetici onayına gönderildi.";
        } else {
            $success = "Değişiklik yapılmadı.";
        }

        // Son durumda kullanıcıyı tekrar DB'den çek
        $stmt = $db->prepare("SELECT * FROM portal_users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        $_SESSION['portal_user'] = $user;
    }
}

/* -------------------------------------------------------------------------- */
/*                   PENDING ve BİLDİRİM KAYITLARINI ÇEK                    */
/* -------------------------------------------------------------------------- */

$pendingChanges = [];
$notifications  = [];

try {
    // Bekleyen (onaylanmamış) kayıtlar
    $stmt = $db->prepare("
        SELECT field, new_value, created_at
        FROM portal_profile_changes
        WHERE user_id = ? AND status = 'pending'
        ORDER BY created_at DESC
    ");
    $stmt->execute([$userId]);
    $pendingChanges = $stmt->fetchAll();
} catch (Exception $e) {
    $pendingChanges = [];
}

try {
    // Onaylanan / reddedilen kayıtlar
    $stmt = $db->prepare("
        SELECT field, old_value, new_value, status, admin_note, created_at, decided_at
        FROM portal_profile_changes
        WHERE user_id = ? AND status IN ('approved','rejected')
        ORDER BY id DESC
        LIMIT 30
    ");
    $stmt->execute([$userId]);
    $notifications = $stmt->fetchAll();
} catch (Exception $e) {
    $notifications = [];
}

// Evrenler listesi
try {
    $uniStmt   = $db->query("SELECT * FROM portal_universes ORDER BY id ASC");
    $universes = $uniStmt->fetchAll();
} catch (Exception $e) {
    $universes = [];
}

require __DIR__ . '/../includes/header.php';
?>

<?php if (isset($user['email_verified']) && $user['email_verified'] == 0): ?>

<div style="
    max-width: 600px;
    margin: 18px auto;
    padding: 18px 22px;
    background: #2d3648;
    border-left: 4px solid #fbbf24;
    color:#eee;
    border-radius: 8px;
    font-size: 14px;
">
    <strong>E-posta doğrulaması bekleniyor.</strong><br>
    Hesabın doğrulanana kadar <strong>hiçbir evrene giriş yapamazsın</strong>.
    Lütfen e-posta kutunu kontrol et veya aşağıdan tekrar doğrulama maili gönder.
</div>

<?php endif; ?>

<?php if (isset($_GET['verify_pending'])): ?>
    <div style="
        max-width: 580px;
        margin: 24px auto 0;
        background: #111726;
        padding: 24px 26px;
        border-radius: 14px;
        text-align: center;
        color: #d2d2d2;
        box-shadow: 0 0 20px rgba(0,0,0,0.4);
        border: 1px solid rgba(59,130,246,0.4);
    ">
        <img src="assets/img/logo-3d.png"
             alt="Novapus"
             style="width: 120px; margin-bottom: 10px; filter: drop-shadow(0 0 10px #4db8ff);">

        <h2 style="font-size: 20px; margin-bottom: 6px; color:#fff;">
            E-postanı Doğrulaman Gerekiyor
        </h2>

        <p style="font-size: 14px; line-height: 1.7; margin-bottom: 12px; color:#cbd5f5;">
            Novapus hesabını aktif etmek için
            <strong>e-posta kutuna gönderilen doğrulama bağlantısına</strong> tıklamalısın.<br>
            Eğer e-posta gelmediyse birkaç dakika içinde tekrar kontrol et.
        </p>

        <?php if (isset($_GET['resent'])): ?>
            <div style="font-size: 13px; color:#4ade80; margin-bottom:8px;">
                ✅ Doğrulama maili tekrar gönderildi!
            </div>
        <?php endif; ?>

        <a href="index.php?page=resend_verify"
           style="
                display:inline-block;
                margin-top: 6px;
                padding: 10px 20px;
                border-radius: 999px;
                background: linear-gradient(90deg,#3b82f6,#22c55e);
                color: #fff;
                text-decoration: none;
                font-size: 14px;
                box-shadow: 0 0 16px rgba(56,189,248,0.7);
           ">
            Doğrulama Mailini Yeniden Gönder
        </a>
    </div>
<?php endif; ?>

<section class="section">


    <h2>Profilim</h2>

    <!-- ÜST PROFİL KARTI -->
    <div class="profile-topcard">
        <div class="ptc-avatar">
            <img src="<?= htmlspecialchars($user['avatar'] ?: 'assets/img/default-avatar.png') ?>" alt="Avatar">
        </div>

        <div class="ptc-info">
            <h2 class="ptc-name"><?= htmlspecialchars($user['username']) ?></h2>
            <div class="ptc-meta">
                <span class="ptc-pill"><?= htmlspecialchars($user['email']) ?></span>
                <span class="ptc-pill">ID: <?= (int)$user['id'] ?></span>
            </div>

            <?php if (!empty($pendingChanges)): ?>
                <div class="ptc-badge">
                    Profil değişikliklerin <strong>onay bekliyor</strong>.
                </div>
            <?php endif; ?>
        </div>
    </div>

    <?php if (!empty($success)): ?>
        <div class="form-success" style="margin-bottom:12px;color:#22c55e;">
            <?= htmlspecialchars($success) ?>
        </div>
    <?php endif; ?>

    <?php if (!empty($message)): ?>
        <div class="form-error" style="margin-bottom:12px;color:#f87171;">
            <?= htmlspecialchars($message) ?>
        </div>
    <?php endif; ?>

    <!-- SEKME BUTONLARI -->
    <div class="profile-tabs">
        <button class="tab-btn active" data-tab="tab-universes">Evren Hesaplarım</button>
        <button class="tab-btn" data-tab="tab-support">Destek</button>
        <button class="tab-btn" data-tab="tab-notifications">Bildirimler</button>
        <button class="tab-btn" data-tab="tab-settings">Ayarlar</button>
    </div>

    <!-- TAB: EVREN HESAPLARIM -->
    <div class="tab-content active" id="tab-universes">
        <div class="panel">
            <h3>Evren Hesaplarım</h3>
            <p style="font-size:12px;color:#9ca3af;margin-bottom:10px;">
                Portal şifren evren şifrinle aynıdır. Portaldan giriş yaptıktan sonra
                herhangi bir evrene girdiğinde, hesabın yoksa otomatik oluşturulur.
            </p>

            <?php if (empty($universes)): ?>
                <p>Şu anda tanımlı evren bulunmuyor.</p>
            <?php else: ?>
                <?php foreach ($universes as $uni): ?>
                    <?php
                        if ($uni['status'] === 'active') {
                            $statusLabel = 'AKTİF';
                            $statusClass = 'uac-status-active';
                        } elseif ($uni['status'] === 'soon') {
                            $statusLabel = 'YAKINDA';
                            $statusClass = 'uac-status-soon';
                        } else {
                            $statusLabel = 'KAPALI';
                            $statusClass = 'uac-status-closed';
                        }
                    ?>
                    <div class="universe-account-card">
                        <div class="uac-header">
                            <div>
                                <h4><?= htmlspecialchars($uni['name']) ?></h4>
                                <div class="uac-sub">
                                    Filo x<?= htmlspecialchars($uni['speed_fleet']) ?> ·
                                    Ekonomi x<?= htmlspecialchars($uni['speed_economy']) ?> ·
                                    Araştırma x<?= htmlspecialchars($uni['speed_research']) ?>
                                </div>
                            </div>
                            <span class="uac-status <?= $statusClass; ?>"><?= $statusLabel; ?></span>
                        </div>

                        <p class="uac-desc">
                            <?= htmlspecialchars($uni['short_description'] ?? 'Bu evren için açıklama bulunmuyor.') ?>
                        </p>

                        <div class="uac-actions">
                            <?php if ($user['email_verified'] == 0): ?>

    <button class="btn btn-disabled btn-xs" disabled>
        Doğrulama Bekleniyor
    </button>

<?php else: ?>

    <?php if ($uni['status'] === 'active'): ?>
        <a href="index.php?page=play&universe=<?= (int)$uni['id'] ?>"
           class="btn btn-primary btn-xs"
           target="_blank">
           Evrene Git
        </a>
    <?php else: ?>
        <button class="btn btn-disabled btn-xs" disabled>Şu anda giriş yapılamaz</button>
    <?php endif; ?>

<?php endif; ?>

                             
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <!-- TAB: DESTEK -->
    <div class="tab-content" id="tab-support">
        <div class="panel">
            <?php
                // Kullanıcı destek listesi bu sayfa içinde gösteriliyor
                require __DIR__ . '/support_list.php';
            ?>
        </div>
    </div>

    <!-- TAB: BİLDİRİMLER -->
    <div class="tab-content" id="tab-notifications">
        <div class="panel">
            <h3>Bildirimler</h3>

            <?php if (empty($pendingChanges) && empty($notifications)): ?>
                <p style="font-size:13px;color:#9ca3af;">
                    Şu anda gösterilecek bildirim bulunmuyor.
                </p>
            <?php endif; ?>

            <?php if (!empty($pendingChanges)): ?>
                <div class="notif-group">
                    <h4 style="font-size:13px;margin-bottom:6px;">Onay Bekleyenler</h4>
                    <?php foreach ($pendingChanges as $row): ?>
                        <div class="notif-item notif-pending">
                            <span class="notif-field">
                                <?php if ($row['field'] === 'username'): ?>
                                    Kullanıcı adı değişikliği
                                <?php elseif ($row['field'] === 'email'): ?>
                                    E-posta değişikliği
                                <?php else: ?>
                                    Avatar değişikliği
                                <?php endif; ?>
                            </span>
                            <span class="notif-value">
                                <?php if ($row['field'] === 'avatar'): ?>
                                    Yeni avatar önizlemesi:
                                    <img src="<?= htmlspecialchars($row['new_value']) ?>" alt="" style="width:32px;height:32px;border-radius:999px;vertical-align:middle;margin-left:4px;">
                                <?php else: ?>
                                    Yeni değer: <strong><?= htmlspecialchars($row['new_value']) ?></strong>
                                <?php endif; ?>
                            </span>
                            <span class="notif-status">Durum: Onay bekliyor</span>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <?php if (!empty($notifications)): ?>
                <div class="notif-group" style="margin-top:12px;">
                    <h4 style="font-size:13px;margin-bottom:6px;">Geçmiş İşlemler</h4>
                    <?php foreach ($notifications as $row): ?>
                        <div class="notif-item <?= $row['status'] === 'approved' ? 'notif-approved' : 'notif-rejected' ?>">
                            <span class="notif-field">
                                <?php if ($row['field'] === 'username'): ?>
                                    Kullanıcı adı değişikliği
                                <?php elseif ($row['field'] === 'email'): ?>
                                    E-posta değişikliği
                                <?php else: ?>
                                    Avatar değişikliği
                                <?php endif; ?>
                            </span>

                            <span class="notif-value">
                                <?php if ($row['field'] === 'avatar'): ?>
                                    <span>Eski avatar</span>
                                    <?php if (!empty($row['old_value'])): ?>
                                        <img src="<?= htmlspecialchars($row['old_value']) ?>" style="width:24px;height:24px;border-radius:999px;vertical-align:middle;margin:0 4px;">
                                    <?php endif; ?>
                                    <span>Yeni avatar</span>
                                    <img src="<?= htmlspecialchars($row['new_value']) ?>" style="width:24px;height:24px;border-radius:999px;vertical-align:middle;margin-left:4px;">
                                <?php else: ?>
                                    <span>Eski: <strong><?= htmlspecialchars($row['old_value']) ?></strong></span>
                                    <span style="margin-left:8px;">Yeni: <strong><?= htmlspecialchars($row['new_value']) ?></strong></span>
                                <?php endif; ?>
                            </span>

                            <span class="notif-status">
                                Durum:
                                <?php if ($row['status'] === 'approved'): ?>
                                    <strong style="color:#22c55e;">Onaylandı</strong>
                                <?php else: ?>
                                    <strong style="color:#f97373;">Reddedildi</strong>
                                <?php endif; ?>
                            </span>

                            <?php if (!empty($row['admin_note'])): ?>
                                <div class="notif-note">
                                    Admin Notu: <?= nl2br(htmlspecialchars($row['admin_note'])) ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

        </div>
    </div>

    <!-- TAB: AYARLAR (HESAP + AVATAR) -->
    <div class="tab-content" id="tab-settings">
        <div class="panel">
            <h3>Ayarlar</h3>

            <p style="font-size:12px;color:#9ca3af;margin-bottom:12px;">
                Kullanıcı adı, e-posta ve avatar değişikliklerin önce <strong>yönetici onayına</strong> gönderilir.
                Şifre değişiklikleri ise <strong>anında uygulanır</strong>.
            </p>

            <form action="" method="post" enctype="multipart/form-data">
                <div class="account-grid">

                    <!-- Kullanıcı adı -->
                    <label class="form-label icon-field user-icon">
                        Kullanıcı Adı
                        <input type="text" name="username" class="form-input"
                               value="<?= htmlspecialchars($user['username']) ?>" required>
                    </label>

                    <!-- Email -->
                    <label class="form-label icon-field email-icon">
                        E-posta
                        <input type="email" name="email" class="form-input"
                               value="<?= htmlspecialchars($user['email']) ?>" required>
                    </label>

                    <!-- Eski Şifre -->
                    <label class="form-label icon-field lock-icon">
                        Eski Şifre (opsiyonel)
                        <input type="password" name="old_password" class="form-input">
                    </label>

                    <!-- Yeni Şifre -->
                    <label class="form-label icon-field lock-icon">
                        Yeni Şifre
                        <input type="password" name="new_password" class="form-input">
                    </label>

                    <!-- Yeni Şifre 2 -->
                    <label class="form-label icon-field lock-icon">
                        Yeni Şifre (Tekrar)
                        <input type="password" name="new_password2" class="form-input">
                    </label>

                </div>

                <div class="avatar-settings-wrapper" style="margin-top:18px;">
                    <h4 style="font-size:13px;margin-bottom:6px;">Avatar</h4>
                    <div class="avatar-panel-inline">
                        <div class="avatar-frame">
                            <img src="<?= htmlspecialchars($user['avatar'] ?: 'assets/img/default-avatar.png') ?>"
                                 class="avatar-img"
                                 style="width:135px;height:135px;border-radius:999px;object-fit:cover;">
                        </div>

                        <div class="avatar-controls">
                            <input type="file" name="avatar" id="avatarUpload" class="form-input" style="padding:6px 8px;">
                            <div style="font-size:11px;color:#9ca3af;margin-top:4px;">
                                Maksimum 2MB, PNG / JPG / JPEG
                            </div>

                            <?php if (!empty($pendingChanges)): ?>
                                <?php foreach ($pendingChanges as $row): ?>
                                    <?php if ($row['field'] === 'avatar'): ?>
                                        <div style="font-size:11px;color:#fbbf24;margin-top:6px;">
                                            Yeni avatar isteğin onay bekliyor.
                                        </div>
                                        <?php break; ?>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <button type="submit" name="save_settings" class="btn btn-primary" style="margin-top:16px;">
                    Değişiklikleri Kaydet
                </button>
            </form>
        </div>
    </div>

</section>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const btns = document.querySelectorAll('.profile-tabs .tab-btn');
    const tabs = document.querySelectorAll('.tab-content');

    btns.forEach(btn => {
        btn.addEventListener('click', function () {
            btns.forEach(b => b.classList.remove('active'));
            this.classList.add('active');

            const id = this.dataset.tab;
            tabs.forEach(t => {
                if (t.id === id) {
                    t.classList.add('active');
                } else {
                    t.classList.remove('active');
                }
            });
        });
    });
});
</script>

<?php require __DIR__ . '/../includes/footer.php'; ?>
